/**
 * Centinel Analytica Azure Functions Protection
 * HTTP Trigger Function for Azure Functions
 *
 * @version 1.0.0
 * @description Bot protection and request validation for Azure Functions
 * @author Centinel Analytica
 */

'use strict';

/*****************************************************************************
 *                                                                           *
 *                       USER CONFIGURATION SECTION                          *
 *                                                                           *
 *               EDIT THESE SETTINGS TO CONFIGURE CENTINEL                   *
 *                                                                           *
 *****************************************************************************/

// ============================================================================
// STEP 1: Centinel API Configuration (REQUIRED)
// ============================================================================

/**
 * Your Centinel secret key
 * Get this from your Centinel Analytica dashboard
 * @required
 */
var CENTINEL_SECRET_KEY = process.env.CENTINEL_SECRET_KEY || '';

// ============================================================================
// STEP 2: Path Protection Configuration (OPTIONAL)
// ============================================================================

/**
 * Protected paths - Only these paths will be validated
 *
 * Leave EMPTY to protect ALL paths (recommended for most use cases)
 *
 * Pattern examples:
 *   '/admin/*'  - Protects /admin/dashboard, /admin/users, etc.
 *   '/api/*'    - Protects /api/users, /api/posts, etc.
 *   '/login'    - Protects exact path only
 *
 * Note: Wildcards (*) match ONE path segment only, not nested paths
 *
 * @type {Array<string>}
 * @default [] (empty = protect all paths)
 */
var CENTINEL_PROTECTED_PATHS = [];

/**
 * Unprotected paths - These paths will SKIP validation entirely
 *
 * Use this to exclude static assets for better performance
 *
 * Pattern examples:
 *   '*.js'   - Excludes ALL JavaScript files at any depth
 *   '*.css'  - Excludes ALL CSS files at any depth
 *   '*.png'  - Excludes ALL PNG images at any depth
 *
 * Extension patterns (*.ext) match files at ANY directory depth:
 *   - /app.js ✅
 *   - /static/js/app.js ✅
 *   - /cdn/assets/bundle.min.js ✅
 *
 * @type {Array<string>}
 * @default Comprehensive list of common static asset extensions
 */
var CENTINEL_UNPROTECTED_PATHS = [
    // Video files
    '*.avi', '*.flv', '*.mka', '*.mkv', '*.mov', '*.mp4', '*.mpeg', '*.mpg',

    // Audio files
    '*.mp3', '*.flac', '*.ogg', '*.ogm', '*.opus', '*.wav', '*.webm',

    // Image files
    '*.webp', '*.bmp', '*.gif', '*.ico', '*.jpeg', '*.jpg', '*.png',
    '*.svg', '*.svgz', '*.swf',

    // Font files
    '*.eot', '*.otf', '*.ttf', '*.woff', '*.woff2',

    // Stylesheets and scripts
    '*.css', '*.less', '*.js', '*.map'
];

// ============================================================================
// STEP 3: Advanced Settings (OPTIONAL)
// ============================================================================

/**
 * Enable detailed logging to Azure Application Insights
 * Useful for troubleshooting and monitoring
 *
 * @type {boolean}
 * @default true
 */
var CENTINEL_LOG_ENABLED = true;

/**
 * Centinel validator API endpoint
 * Only change this if you're using a custom validator endpoint
 *
 * @type {string}
 * @default 'validator.centinelanalytica.com'
 */
var CENTINEL_VALIDATOR_URL = process.env.CENTINEL_VALIDATOR_URL || 'validator.centinelanalytica.com';

/**
 * Validator API timeout in milliseconds
 * Requests taking longer than this will fail-open (allow request)
 *
 * @type {number}
 * @default 100 (0.1 seconds)
 */
var CENTINEL_TIMEOUT = Number(process.env.CENTINEL_TIMEOUT || 100);

/**
 * Backend URL for reverse proxy mode (OPTIONAL)
 *
 * Two deployment modes are supported:
 *
 * 1. REVERSE PROXY MODE (set CENTINEL_BACKEND_URL):
 *    Function validates requests and proxies allowed traffic to your backend.
 *    Flow: Client → Front Door → Function → Backend
 *
 * 2. VALIDATION-ONLY MODE (leave CENTINEL_BACKEND_URL empty):
 *    Function only validates and returns allow/block decision.
 *    Use this when deploying as a Front Door extension where Front Door
 *    handles routing to your backend based on the function's response.
 *    Flow: Client → Function → (allowed: 200 OK, blocked: challenge page)
 *
 * @type {string}
 * @default '' (validation-only mode)
 */
var CENTINEL_BACKEND_URL = process.env.CENTINEL_BACKEND_URL || '';

/**
 * HTTP status code for redirect/challenge responses
 * Default is 429 to allow the challenge page to render properly
 * Some integrations may require a different status code (e.g., 202, 429)
 *
 * @type {number}
 * @default 429
 */
var CENTINEL_REDIRECT_STATUS = Number(process.env.CENTINEL_REDIRECT_STATUS || 429);

/**
 * HTTP status code for block responses
 * Default is 403 (Forbidden)
 * Some integrations may require a different status code (e.g., 401, 429)
 *
 * @type {number}
 * @default 403
 */
var CENTINEL_BLOCK_STATUS = Number(process.env.CENTINEL_BLOCK_STATUS || 403);

/*****************************************************************************
 *                                                                           *
 *                      END OF CONFIGURATION SECTION                         *
 *                                                                           *
 *                 DO NOT EDIT ANYTHING BELOW THIS LINE                      *
 *                                                                           *
 *   The code below is the Centinel module implementation and should         *
 *   not be modified. Editing this code may break functionality and          *
 *   prevent the module from working correctly.                              *
 *                                                                           *
 *****************************************************************************/

// ============================================================================
// Module Constants (DO NOT EDIT)
// ============================================================================

const centinelValidatorPath = '/validate';
const centinelContentType = 'application/json';
const centinelModuleName = 'Azure Functions';
const centinelModuleVersion = '1.0.0';

// ============================================================================
// Internal State Variables (DO NOT EDIT)
// ============================================================================

// Cache for compiled regex patterns (initialized at module load)
let protectedPathRegexes = [];
let unprotectedPathRegexes = [];

// Exponential backoff configuration
const BACKOFF_INITIAL_DELAY = 1000;      // 1 second
const BACKOFF_MAX_DELAY = 300000;        // 5 minutes
const BACKOFF_MULTIPLIER = 2;

// Backoff state (shared across function invocations in same instance)
let backoffFailureCount = 0;
let backoffLastFailureTime = 0;
let backoffCurrentDelay = 0;

// Configuration object (DO NOT EDIT)
const centinelConfiguration = {
    secretKey: CENTINEL_SECRET_KEY,
    validatorUrl: CENTINEL_VALIDATOR_URL,
    timeout: CENTINEL_TIMEOUT,
    protectedPaths: CENTINEL_PROTECTED_PATHS,
    unprotectedPaths: CENTINEL_UNPROTECTED_PATHS,
    logEnabled: CENTINEL_LOG_ENABLED,
    backendUrl: CENTINEL_BACKEND_URL
};

// Required Node.js modules
const https = require('https');
const http = require('http');

// Create persistent HTTPS agent for connection pooling and keep-alive
// This agent is reused across function invocations in the same instance
const httpsAgent = new https.Agent({
    keepAlive: true,
    keepAliveMsecs: 1000,           // Send keep-alive probes every 1 second
    maxSockets: 50,                  // Max concurrent connections per host
    maxFreeSockets: 10,              // Max idle connections to keep open
    timeout: 60000,                  // Socket timeout (60 seconds)
    scheduling: 'lifo'               // Use most recently used socket first
});

// ============================================================================
// Module Implementation (DO NOT EDIT)
// ============================================================================

/**
 * Converts a path pattern to a safe regex
 * Handles both simple wildcards (*.js) and path wildcards (/path/*)
 *
 * @private
 * @param {string} pattern - Path pattern to compile
 * @returns {RegExp} Compiled regex pattern
 */
function compilePathPattern(pattern) {
    // Check if pattern starts with * (e.g., *.js, *.css)
    // These should match any path ending with the extension
    if (pattern.startsWith('*')) {
        // Escape special regex characters except *
        const escaped = pattern
            .substring(1) // Remove leading *
            .replace(/[.+?^${}()|[\]\\]/g, '\\$&');

        // Match any path ending with the pattern
        // e.g., *.js becomes /.*\.js$/ which matches /path/to/file.js
        const regexPattern = '.*' + escaped + '$';
        return new RegExp(regexPattern);
    }

    // For patterns with paths (e.g., /admin/*, /api/*)
    // Use [^/]* to match within a single path segment
    const escaped = pattern
        .replace(/[.+?^${}()|[\]\\]/g, '\\$&')
        .replace(/\*/g, '%%%WILDCARD%%%');

    // Replace wildcard placeholder with safe pattern
    // [^/]* matches any character except / (prevents backtracking across path segments)
    const regexPattern = '^' + escaped.replace(/%%%WILDCARD%%%/g, '[^/]*') + '$';

    return new RegExp(regexPattern);
}

/**
 * Precompile all path patterns at module load time
 * This improves performance by avoiding regex compilation on every request
 *
 * @private
 */
function compilePathPatterns() {
    try {
        // Compile protected paths
        if (CENTINEL_PROTECTED_PATHS && Array.isArray(CENTINEL_PROTECTED_PATHS)) {
            protectedPathRegexes = CENTINEL_PROTECTED_PATHS
                .filter(pattern => pattern && typeof pattern === 'string')
                .map(pattern => ({
                    pattern: pattern,
                    regex: compilePathPattern(pattern)
                }));
        }

        // Compile unprotected paths
        if (CENTINEL_UNPROTECTED_PATHS && Array.isArray(CENTINEL_UNPROTECTED_PATHS)) {
            unprotectedPathRegexes = CENTINEL_UNPROTECTED_PATHS
                .filter(pattern => pattern && typeof pattern === 'string')
                .map(pattern => ({
                    pattern: pattern,
                    regex: compilePathPattern(pattern)
                }));
        }

        console.log('[Centinel] Path patterns compiled', {
            protected: protectedPathRegexes.length,
            unprotected: unprotectedPathRegexes.length
        });
    } catch (error) {
        console.error('[Centinel] Error compiling path patterns:', error.message);
        // Set to empty arrays on error to allow fail-safe operation
        protectedPathRegexes = [];
        unprotectedPathRegexes = [];
    }
}

// Compile patterns at module load time
compilePathPatterns();

// ============================================================================
// Public API (DO NOT EDIT)
// ============================================================================

/**
 * Azure Functions HTTP trigger handler
 * This is the main entry point called by Azure for each HTTP request
 *
 * @public
 * @param {Object} context - Azure Functions context object
 * @param {Object} req - HTTP request object
 * @returns {Promise<void>}
 */
module.exports = async function (context, req) {
    try {
        log(context, 'Processing request', {
            url: req.url,
            method: req.method,
            path: req.params?.path || req.url
        });

        // Validate configuration
        if (!centinelConfiguration.secretKey) {
            log(context, 'Missing secret key, allowing request');
            return handlePassthrough(context, req);
        }

        // Get request path from URL
        const requestPath = getRequestPath(req);

        // Check if path is protected
        if (!isPathProtected(requestPath)) {
            log(context, 'Path not protected, allowing request');
            return handlePassthrough(context, req);
        }

        // Check if we should skip validator call due to backoff
        if (shouldSkipValidatorCall()) {
            log(context, 'Skipped validation due to backoff, allowing request');
            return handlePassthrough(context, req);
        }

        // Extract Centinel cookie
        const centinelCookie = extractCentinelCookie(req.headers);
        if (!centinelCookie) {
            log(context, 'No Centinel cookie found');
        }

        // Get client IP address
        const clientIp = getClientIp(req);

        // Build full URL
        const fullUrl = getFullUrl(req);

        log(context, 'Request details', {
            ip: clientIp,
            path: requestPath,
            url: fullUrl
        });

        // Prepare validation data
        const validationData = {
            url: fullUrl,
            method: req.method || 'GET',
            ip: clientIp,
            headers: req.headers || {},
            referrer: req.headers?.referer || req.headers?.referrer || '',
            cookie: centinelCookie || ''
        };

        // Make validation request
        const validationResponse = await makeValidationRequest(validationData);

        // Validate response structure
        if (!validationResponse || typeof validationResponse !== 'object') {
            log(context, 'Invalid validation response structure, allowing request');
            return handlePassthrough(context, req);
        }

        const decision = validationResponse.decision || 'allow';
        const cookies = validationResponse.cookies;

        log(context, 'Validation response received', {
            decision,
            hasCookies: !!cookies
        });

        // Handle decision
        switch (decision) {
            case 'block':
                log(context, 'Blocking request', { decision });
                return createBlockResponse(context, validationResponse.response_html, 'block', cookies);

            case 'redirect':
                log(context, 'Redirecting to challenge page', { decision });
                return createBlockResponse(context, validationResponse.response_html, 'redirect', cookies);

            case 'allow':
            case 'not_matched':
                log(context, 'Allowing request', { decision });
                return handleAllowResponse(context, req, cookies);

            default:
                log(context, 'Unknown decision, allowing request', { decision });
                return handleAllowResponse(context, req, cookies);
        }
    } catch (error) {
        log(context, 'Unexpected error during validation, allowing request', {
            error: error.message,
            stack: error.stack
        });
        return handlePassthrough(context, req);
    }
};

// ============================================================================
// Helper Functions (DO NOT EDIT)
// ============================================================================

/**
 * Log messages to Azure Application Insights (if logging is enabled)
 *
 * @private
 * @param {Object} context - Azure Functions context
 * @param {string} message - Log message
 * @param {Object|null} data - Optional data to log
 */
function log(context, message, data = null) {
    if (centinelConfiguration.logEnabled) {
        const logMessage = `[Centinel] ${message}`;
        if (data) {
            context.log(logMessage, data);
        } else {
            context.log(logMessage);
        }
    }
}

/**
 * Record a validator API failure and update backoff state
 *
 * @private
 */
function recordValidatorFailure() {
    backoffFailureCount++;
    backoffLastFailureTime = Date.now();

    // Calculate new backoff delay with exponential increase
    if (backoffFailureCount === 1) {
        backoffCurrentDelay = BACKOFF_INITIAL_DELAY;
    } else {
        backoffCurrentDelay = Math.min(
            backoffCurrentDelay * BACKOFF_MULTIPLIER,
            BACKOFF_MAX_DELAY
        );
    }

    console.log('[Centinel] Validator failure recorded, backoff updated', {
        failureCount: backoffFailureCount,
        currentDelay: backoffCurrentDelay
    });
}

/**
 * Record a validator API success and reset backoff state
 *
 * @private
 */
function recordValidatorSuccess() {
    if (backoffFailureCount > 0) {
        console.log('[Centinel] Validator success, backoff reset', {
            previousFailureCount: backoffFailureCount
        });
    }

    backoffFailureCount = 0;
    backoffLastFailureTime = 0;
    backoffCurrentDelay = 0;
}

/**
 * Check if we should skip the validator API call due to backoff
 * Returns true if we should skip the call (still in backoff period)
 *
 * @private
 * @returns {boolean} True if validator call should be skipped
 */
function shouldSkipValidatorCall() {
    // No failures, proceed normally
    if (backoffFailureCount === 0) {
        return false;
    }

    // Check if backoff period has expired
    const timeSinceLastFailure = Date.now() - backoffLastFailureTime;

    if (timeSinceLastFailure < backoffCurrentDelay) {
        console.log('[Centinel] Skipping validator call (in backoff period)', {
            failureCount: backoffFailureCount,
            currentDelay: backoffCurrentDelay,
            timeSinceLastFailure,
            remainingDelay: backoffCurrentDelay - timeSinceLastFailure
        });
        return true;
    }

    // Backoff period expired, allow retry
    console.log('[Centinel] Backoff period expired, retrying validator call', {
        failureCount: backoffFailureCount,
        timeSinceLastFailure
    });
    return false;
}

/**
 * Check if a request path should be protected (validated)
 *
 * @private
 * @param {string} requestPath - The request URI path
 * @returns {boolean} True if path should be protected
 */
function isPathProtected(requestPath) {
    try {
        // First check if path is in unprotected list (has priority)
        if (unprotectedPathRegexes.length > 0) {
            const isUnprotected = unprotectedPathRegexes.some(({ regex, pattern }) => {
                try {
                    return regex.test(requestPath);
                } catch (error) {
                    console.log('[Centinel] Error testing unprotected path pattern', {
                        pattern,
                        path: requestPath,
                        error: error.message
                    });
                    return false;
                }
            });

            if (isUnprotected) {
                return false; // Path is unprotected, skip validation
            }
        }

        // Then check if path is in protected list
        if (protectedPathRegexes.length === 0) {
            return true; // Protect all paths if none specified
        }

        return protectedPathRegexes.some(({ regex, pattern }) => {
            try {
                return regex.test(requestPath);
            } catch (error) {
                console.log('[Centinel] Error testing protected path pattern', {
                    pattern,
                    path: requestPath,
                    error: error.message
                });
                return false;
            }
        });
    } catch (error) {
        console.log('[Centinel] Error in isPathProtected, defaulting to protected', {
            path: requestPath,
            error: error.message
        });
        return true; // Default to protected on error (fail-safe)
    }
}

/**
 * Extract request path from Azure Functions request object
 *
 * @private
 * @param {Object} req - Azure Functions request object
 * @returns {string} Request path
 */
function getRequestPath(req) {
    try {
        // Try to get path from various sources
        if (req.params && req.params.path) {
            return '/' + req.params.path;
        }

        if (req.url) {
            const parsedUrl = new URL(req.url, 'https://localhost');
            return parsedUrl.pathname || '/';
        }

        return '/';
    } catch (error) {
        console.log('[Centinel] Error extracting request path', { error: error.message });
        return '/';
    }
}

/**
 * Get full URL from Azure Functions request object
 *
 * @private
 * @param {Object} req - Azure Functions request object
 * @returns {string} Full URL
 */
function getFullUrl(req) {
    try {
        const forwardedProto = req.headers['x-forwarded-proto'];
        const protocol = (Array.isArray(forwardedProto) ? forwardedProto[0] : forwardedProto || 'https')
            .toString()
            .split(',')[0]
            .trim() || 'https';

        const host =
            req.headers['x-forwarded-host'] ||
            req.headers['x-original-host'] ||
            req.headers.host ||
            'unknown-host';
        const path = getRequestPath(req);

        // Get query string
        let queryString = '';
        if (req.url) {
            const parsedUrl = new URL(req.url, 'https://localhost');
            queryString = parsedUrl.search || '';
        }

        return `${protocol}://${host}${path}${queryString}`;
    } catch (error) {
        console.log('[Centinel] Error building full URL', { error: error.message });
        return 'https://unknown-host/';
    }
}

/**
 * Get client IP address from Azure Functions request
 *
 * @private
 * @param {Object} req - Azure Functions request object
 * @returns {string} Client IP address
 */
function getClientIp(req) {
    try {
        // Azure Functions provides the client IP in various headers
        return req.headers['x-forwarded-for']?.split(',')[0]?.trim() ||
               req.headers['x-azure-clientip'] ||
               req.headers['x-client-ip'] ||
               req.headers['x-real-ip'] ||
               'unknown-ip';
    } catch (error) {
        console.log('[Centinel] Error extracting client IP', { error: error.message });
        return 'unknown-ip';
    }
}

/**
 * Extract Centinel cookie from request headers
 *
 * @private
 * @param {Object} headers - Request headers
 * @returns {string|null} Cookie value or null if not found
 */
function extractCentinelCookie(headers) {
    try {
        const cookieHeader = headers.cookie;
        if (!cookieHeader || typeof cookieHeader !== 'string') return null;

        const cookies = cookieHeader.split(';');
        for (const cookie of cookies) {
            const trimmedCookie = cookie.trim();
            if (!trimmedCookie) continue;

            const [name, value] = trimmedCookie.split('=');
            if (name && name.trim() === '_centinel') {
                return value || '';
            }
        }
        return null;
    } catch (error) {
        console.log('[Centinel] Error extracting Centinel cookie', { error: error.message });
        return null;
    }
}

/**
 * Make validation request to Centinel API
 *
 * @private
 * @param {Object} validationData - Request data to validate
 * @returns {Promise<Object>} Validation response
 */
function makeValidationRequest(validationData) {
    return new Promise((resolve) => {
        const startTime = Date.now();
        const postData = JSON.stringify(validationData);

        const options = {
            hostname: centinelConfiguration.validatorUrl,
            port: 443,
            path: centinelValidatorPath,
            method: 'POST',
            headers: {
                'Content-Type': centinelContentType,
                'Content-Length': Buffer.byteLength(postData),
                'x-api-key': centinelConfiguration.secretKey,
                'x-origin-module': centinelModuleName,
                'x-origin-version': centinelModuleVersion
            },
            timeout: centinelConfiguration.timeout,
            agent: httpsAgent
        };

        const req = https.request(options, (res) => {
            let data = '';

            res.on('data', (chunk) => {
                data += chunk;
            });

            res.on('end', () => {
                const duration = Date.now() - startTime;
                try {
                    // Check for successful HTTP status codes (2xx)
                    if (res.statusCode < 200 || res.statusCode >= 300) {
                        console.log('[Centinel] Validation API returned error status', {
                            statusCode: res.statusCode,
                            data: data.substring(0, 200), // Log first 200 chars
                            validatorDurationMs: duration
                        });
                        recordValidatorFailure();
                        resolve({ decision: 'allow' });
                        return;
                    }

                    // Validate response is not empty
                    if (!data || data.trim().length === 0) {
                        console.log('[Centinel] Validation API returned empty response', {
                            validatorDurationMs: duration
                        });
                        recordValidatorFailure();
                        resolve({ decision: 'allow' });
                        return;
                    }

                    // Parse JSON response
                    const response = JSON.parse(data);

                    // Validate response structure
                    if (typeof response !== 'object' || response === null) {
                        console.log('[Centinel] Validation API returned invalid response structure', {
                            validatorDurationMs: duration
                        });
                        recordValidatorFailure();
                        resolve({ decision: 'allow' });
                        return;
                    }

                    // Check if API returned an error response
                    if (response.success === false) {
                        console.log('[Centinel] Validation API error', {
                            status: response.status,
                            message: response.message,
                            validatorDurationMs: duration
                        });
                        recordValidatorFailure();
                        resolve({ decision: 'allow' });
                        return;
                    }

                    // Validate decision field exists
                    if (!response.decision) {
                        console.log('[Centinel] Validation API response missing decision field', {
                            validatorDurationMs: duration
                        });
                        recordValidatorFailure();
                        resolve({ decision: 'allow' });
                        return;
                    }

                    // Success - record it and return response with timing
                    console.log('[Centinel] Validator response received', {
                        decision: response.decision,
                        validatorDurationMs: duration
                    });
                    recordValidatorSuccess();
                    resolve(response);
                } catch (error) {
                    console.log('[Centinel] Failed to parse validation response', {
                        error: error.message,
                        data: data.substring(0, 200), // Log first 200 chars
                        validatorDurationMs: duration
                    });
                    recordValidatorFailure();
                    resolve({ decision: 'allow' });
                }
            });
        });

        req.on('error', (error) => {
            const duration = Date.now() - startTime;
            console.log('[Centinel] Validation request error', {
                error: error.message,
                code: error.code,
                validatorDurationMs: duration
            });
            recordValidatorFailure();
            resolve({ decision: 'allow' });
        });

        req.on('timeout', () => {
            const duration = Date.now() - startTime;
            console.log('[Centinel] Validation request timeout', {
                timeout: centinelConfiguration.timeout,
                validatorDurationMs: duration
            });
            req.destroy();
            recordValidatorFailure();
            resolve({ decision: 'allow' });
        });

        // Handle unexpected errors during request write
        try {
            req.write(postData);
            req.end();
        } catch (error) {
            const duration = Date.now() - startTime;
            console.log('[Centinel] Error sending validation request', {
                error: error.message,
                validatorDurationMs: duration
            });
            recordValidatorFailure();
            resolve({ decision: 'allow' });
        }
    });
}

/**
 * Convert cookie objects from validator API to Set-Cookie header strings
 *
 * @private
 * @param {Array|null} cookies - Array of cookie objects from validator
 * @returns {Array|null} Array of Set-Cookie header strings or null
 */
function convertCookiesToHeaders(cookies) {
    if (!cookies || !Array.isArray(cookies) || cookies.length === 0) {
        return null;
    }

    try {
        return cookies.map(cookie => {
            if (typeof cookie === 'string') {
                // Already a cookie string, return as-is
                return cookie;
            }

            // Convert cookie object to Set-Cookie header format
            let cookieString = `${cookie.name}=${cookie.value}`;

            if (cookie.domain) {
                cookieString += `; Domain=${cookie.domain}`;
            }

            if (cookie.path) {
                cookieString += `; Path=${cookie.path}`;
            }

            if (cookie.expires) {
                cookieString += `; Expires=${cookie.expires}`;
            }

            if (cookie.maxAge) {
                cookieString += `; Max-Age=${cookie.maxAge}`;
            }

            if (cookie.secure) {
                cookieString += '; Secure';
            }

            if (cookie.httpOnly) {
                cookieString += '; HttpOnly';
            }

            if (cookie.sameSite) {
                cookieString += `; SameSite=${cookie.sameSite}`;
            }

            return cookieString;
        });
    } catch (error) {
        console.log('[Centinel] Error converting cookies to headers', { error: error.message });
        return null;
    }
}

/**
 * Handle allow response with optional cookies
 * Either passes through to backend or returns success response
 *
 * @private
 * @param {Object} context - Azure Functions context
 * @param {Object} req - Request object
 * @param {Array|null} cookies - Cookies to set from validator response
 * @returns {Promise<void>}
 */
async function handleAllowResponse(context, req, cookies) {
    const cookieHeaders = convertCookiesToHeaders(cookies);
    return handlePassthrough(context, req, cookieHeaders);
}

/**
 * Handle passthrough to backend application
 * Proxies the request to the backend URL and returns the response
 *
 * @private
 * @param {Object} context - Azure Functions context
 * @param {Object} req - Request object
 * @param {Array|null} additionalCookies - Additional cookies to set
 * @returns {Promise<void>}
 */
async function handlePassthrough(context, req, additionalCookies = null) {
    // If no backend URL configured, run in validation-only mode
    if (!centinelConfiguration.backendUrl) {
        log(context, 'Validation-only mode: request allowed');

        const headers = {
            'Content-Type': 'application/json',
            'X-Centinel-Decision': 'allow',
            'Cache-Control': 'no-store, no-cache, must-revalidate'
        };

        // Add cookies if provided
        if (additionalCookies && additionalCookies.length > 0) {
            headers['Set-Cookie'] = additionalCookies;
        }

        context.res = {
            status: 200,
            headers,
            body: JSON.stringify({
                success: true,
                decision: 'allow',
                message: 'Request validated successfully'
            })
        };
        return;
    }

    // Proxy request to backend using fetch API
    try {
        const requestPath = getRequestPath(req);

        // Get query string
        let queryString = '';
        if (req.url) {
            const parsedUrl = new URL(req.url, 'https://localhost');
            queryString = parsedUrl.search || '';
        }

        const targetUrl = centinelConfiguration.backendUrl + requestPath + queryString;

        // Prepare headers (remove hop-by-hop headers)
        const outgoingHeaders = {};
        const hopByHop = ['connection', 'keep-alive', 'proxy-authenticate', 'proxy-authorization',
                          'te', 'trailers', 'transfer-encoding', 'upgrade', 'host'];

        for (const [key, value] of Object.entries(req.headers || {})) {
            if (!hopByHop.includes(key.toLowerCase())) {
                outgoingHeaders[key] = value;
            }
        }

        log(context, 'Starting proxy request', { targetUrl, method: req.method });

        // Build fetch options
        const fetchOptions = {
            method: req.method || 'GET',
            headers: outgoingHeaders
        };

        // Add body for non-GET/HEAD requests
        const method = String(req.method || 'GET').toUpperCase();
        if (!['GET', 'HEAD', 'OPTIONS'].includes(method)) {
            if (req.rawBody) {
                fetchOptions.body = req.rawBody;
            } else if (req.body) {
                fetchOptions.body = typeof req.body === 'string' ? req.body : JSON.stringify(req.body);
            }
        }

        // Make the request
        const response = await fetch(targetUrl, fetchOptions);

        log(context, 'Proxy response received', {
            status: response.status,
            statusText: response.statusText
        });

        // Get response body as text
        const bodyText = await response.text();

        // Build response headers
        const responseHeaders = {};
        response.headers.forEach((value, key) => {
            responseHeaders[key] = value;
        });

        // Add additional cookies if provided
        if (additionalCookies && additionalCookies.length > 0) {
            const existing = responseHeaders['set-cookie'];
            const existingCookies = existing ? (Array.isArray(existing) ? existing : [existing]) : [];
            responseHeaders['set-cookie'] = [...existingCookies, ...additionalCookies];
        }

        context.res = {
            status: response.status,
            headers: responseHeaders,
            body: bodyText
        };
    } catch (error) {
        log(context, 'Proxy request error', {
            error: error.message,
            stack: error.stack
        });

        context.res = {
            status: 502,
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({
                success: false,
                message: 'Backend service unavailable',
                error: error.message
            })
        };
    }
}

/**
 * Create block/challenge response for Azure Functions
 *
 * @private
 * @param {Object} context - Azure Functions context
 * @param {string} responseHtml - Base64 encoded HTML response from validator
 * @param {string} decisionType - Type of decision ('block' or 'redirect')
 * @param {Array|null} cookies - Cookies to set from validator response
 * @returns {void}
 */
function createBlockResponse(context, responseHtml, decisionType = 'block', cookies = null) {
    let htmlContent;

    if (responseHtml) {
        try {
            // Decode base64 response HTML
            let decodedHtml = Buffer.from(responseHtml, 'base64').toString('utf-8');

            // Try to decode URI components if present
            try {
                decodedHtml = decodeURIComponent(decodedHtml);
            } catch (e) {
                // HTML is not URL-encoded, use as is
                console.log('[Centinel] Response HTML not URI-encoded, using as-is');
            }

            htmlContent = decodedHtml;
        } catch (error) {
            // If base64 decoding fails, log and use default response
            console.log('[Centinel] Failed to decode response HTML, using default', { error: error.message });
            htmlContent = null;
        }
    }

    // Use default HTML if no valid response HTML
    if (!htmlContent) {
        const title = decisionType === 'redirect' ? 'Verification Required' : 'Access Denied';
        const heading = decisionType === 'redirect' ? 'Verification Required' : 'Access Denied';
        const message = decisionType === 'redirect'
            ? 'Please complete the verification challenge to continue.'
            : 'Your request has been blocked by our security system.';

        htmlContent = `
            <!DOCTYPE html>
            <html>
            <head>
                <title>${title}</title>
                <meta charset="utf-8">
                <style>
                    body { font-family: Arial, sans-serif; text-align: center; margin-top: 100px; }
                    .container { max-width: 500px; margin: 0 auto; }
                    h1 { color: ${decisionType === 'redirect' ? '#1976d2' : '#d32f2f'}; }
                    p { color: #666; line-height: 1.6; }
                </style>
            </head>
            <body>
                <div class="container">
                    <h1>${heading}</h1>
                    <p>${message}</p>
                    <p>If you believe this is an error, please contact support.</p>
                </div>
            </body>
            </html>
        `;
    }

    // Use configurable status codes for redirects (challenge pages) and blocks
    const status = decisionType === 'redirect' ? CENTINEL_REDIRECT_STATUS : CENTINEL_BLOCK_STATUS;

    const headers = {
        'Content-Type': 'text/html; charset=utf-8',
        'Cache-Control': 'no-store, no-cache, must-revalidate',
        'X-Centinel-Decision': decisionType
    };

    // Add cookies to response if provided
    const cookieHeaders = convertCookiesToHeaders(cookies);
    if (cookieHeaders && cookieHeaders.length > 0) {
        headers['Set-Cookie'] = cookieHeaders;
        console.log('[Centinel] Added cookies to block/redirect response', {
            cookieCount: cookieHeaders.length
        });
    }

    context.res = {
        status,
        headers,
        body: htmlContent
    };
}
