-- Centinel Bot Protection - Apache Handler
-- This file is referenced by LuaHookAccessChecker in httpd.conf
-- It initializes the module on first load and delegates to the access handler.

-- Module is loaded once per Apache child process (prefork MPM)
local centinel = require("centinel-apache")

-- Initialize on first load with environment variables
local initialized = false

-- Helper: read a config value from multiple sources
local function get_env(r, name)
    -- 1. Apache subprocess_env (SetEnv / PassEnv)
    if r.subprocess_env then
        local val = r.subprocess_env[name]
        if val and val ~= "" then return val end
    end
    -- 2. OS environment (available to the httpd process)
    local val = os.getenv(name)
    if val and val ~= "" then return val end
    return nil
end

function access_check(r)
    if not initialized then
        local secret_key = get_env(r, "CENTINEL_SECRET_KEY")
        local validator_url = get_env(r, "CENTINEL_VALIDATOR_URL")
        local debug_mode = get_env(r, "CENTINEL_DEBUG")

        centinel.init({
            secret_key = secret_key,
            validator_url = validator_url or "https://validator.centinelanalytica.com/validate",
            debug = (debug_mode == "true" or debug_mode == "1"),
        })

        r:notice("[Centinel] Module initialized (v" .. centinel.get_version() .. ")")
        r:notice("[Centinel] Validator URL: " .. (validator_url or "default"))
        r:notice("[Centinel] Secret key: " .. (secret_key and "configured" or "NOT SET"))
        initialized = true
    end

    return centinel.access_handler(r)
end
