// -------------------------------------------Please DO NOT change below---------------------------------------- //
import { httpRequest } from "http-request";
import { SetCookie } from "cookies";

const CENTINEL_INTERNAL_VALIDATE_PATH = "/centinel-validate";
const CENTINEL_COOKIE_NAME = "_centinel";

const DEFAULT_TIMEOUT_MS = 300;

// Default static asset exclusions (same spirit as Centinel Cloudflare Worker)
const DEFAULT_PROTECTED_PATHS_EXCLUSION =
  /\.(avi|avif|bmp|css|eot|flac|flv|gif|gz|ico|jpeg|jpg|js|json|less|map|mka|mkv|mov|mp3|mp4|mpeg|mpg|ogg|ogm|opus|otf|png|svg|svgz|swf|ttf|wav|webm|webp|woff|woff2|xml|zip)$/i;

// Exponential backoff state (shared within EW instance)
let backoffFailureCount = 0;
let backoffLastFailureTime = 0;
let backoffCurrentDelay = 0;

const BACKOFF_INITIAL_DELAY = 1000; // 1s
const BACKOFF_MAX_DELAY = 300000; // 5m
const BACKOFF_MULTIPLIER = 2;

function getHeaderValue(httpMessage, key) {
  if (httpMessage == null || key == null) {
    return "";
  }

  const val = httpMessage.getHeader(key);
  if (val != null && val.length > 0) {
    return val[0];
  }

  return "";
}

function storeCentinelLogHeader(request, value) {
  try {
    request.setVariable("PMUSER_EW_CENTINEL_LOG_HEADER", value);
  } catch (e) {}
}

function appendCentinelLog(request, message) {
  try {
    const current = request.getVariable("PMUSER_EW_CENTINEL_LOG_HEADER") || "";
    const next = current ? `${current} - ${message}` : message;
    request.setVariable("PMUSER_EW_CENTINEL_LOG_HEADER", next);
  } catch (e) {}
}

function toRegExp(pattern) {
  if (pattern == null) return null;

  const str = String(pattern).trim();
  if (!str) return null;

  try {
    // Accept either raw pattern (e.g. "/api/") or "regex literal" style ("/api/i")
    if (str.startsWith("/") && str.lastIndexOf("/") > 0) {
      const lastSlash = str.lastIndexOf("/");
      const body = str.slice(1, lastSlash);
      const flags = str.slice(lastSlash + 1);
      return new RegExp(body, flags);
    }

    return new RegExp(str);
  } catch (e) {
    return null;
  }
}

function shouldProtectPath(requestPath, inclusionRe, exclusionRe) {
  if (!requestPath) return true;

  if (exclusionRe && exclusionRe.test(requestPath)) {
    return false;
  }

  if (inclusionRe && !inclusionRe.test(requestPath)) {
    return false;
  }

  return true;
}

function parseRequestUrlParts(request) {
  // Akamai may expose request.url as a path or full URL depending on context.
  const raw = request.url || "";

  // Full URL case (avoid relying on global URL()).
  if (raw.startsWith("http://") || raw.startsWith("https://")) {
    const match = raw.match(/^https?:\/\/[^/]+(\/[^?#]*)?/i);
    const pathname = match && match[1] ? match[1] : "/";
    return { fullUrl: raw, path: pathname };
  }

  // Path-only case.
  const path = raw.startsWith("/") ? raw : `/${raw}`;
  const fullUrl = `${request.scheme}://${request.host}${path}`;
  return { fullUrl, path: path.split("?")[0] || "/" };
}

function getCentinelCookieValue(request) {
  const cookieHeader = request.getHeader("Cookie");
  if (!cookieHeader || cookieHeader.length === 0) return "";

  // Akamai returns an array of header values; use the first.
  const raw = cookieHeader[0] || "";
  if (!raw) return "";

  // Take the FIRST occurrence if duplicated.
  const parts = raw.split(";");
  for (let i = 0; i < parts.length; i++) {
    const part = parts[i].trim();
    if (!part) continue;

    const eqIdx = part.indexOf("=");
    if (eqIdx <= 0) continue;

    const key = part.slice(0, eqIdx).trim();
    if (key !== CENTINEL_COOKIE_NAME) continue;

    const val = part.slice(eqIdx + 1);
    try {
      return decodeURIComponent(val);
    } catch (e) {
      return val;
    }
  }

  return "";
}

function joinHeaderValues(values) {
  if (values == null) return "";
  if (Array.isArray(values)) return values.join(", ");
  return String(values);
}

function buildHeadersObjectExcludingCookie(request) {
  const headers = request.getHeaders() || {};
  const headerKeys = Object.keys(headers);

  const out = {};
  for (let i = 0; i < headerKeys.length; i++) {
    const key = headerKeys[i];
    if (key == null) continue;

    if (String(key).toLowerCase() === "cookie") {
      continue;
    }

    const v = headers[key];
    const joined = joinHeaderValues(v);

    // Keep header even if empty ("all headers no matter what")
    out[key] = joined;
  }

  return out;
}

function looksLikeIp(ip) {
  if (!ip) return false;
  const s = String(ip).trim();
  if (!s) return false;

  // Very lightweight validation: IPv4 or IPv6-ish.
  const ipv4 = /^\d{1,3}(?:\.\d{1,3}){3}$/;
  const ipv6 = /^[0-9a-fA-F:]+$/;
  return ipv4.test(s) || ipv6.test(s);
}

function getBestClientIp(request) {
  // Prefer Akamai-provided variable if present.
  try {
    const varIp = request.getVariable("PMUSER_EW_CLIENT_IP");
    if (looksLikeIp(varIp)) return String(varIp).trim();
  } catch (e) {}

  const trueClientIp = getHeaderValue(request, "True-Client-IP");
  if (looksLikeIp(trueClientIp)) return trueClientIp.trim();

  const xRealIp = getHeaderValue(request, "X-Real-IP");
  if (looksLikeIp(xRealIp)) return xRealIp.trim();

  const xForwardedFor = getHeaderValue(request, "X-Forwarded-For");
  if (xForwardedFor) {
    const first = xForwardedFor.split(",")[0].trim();
    if (looksLikeIp(first)) return first;
  }

  return "unknown-ip";
}

function recordValidatorFailure(request, reason) {
  backoffFailureCount++;
  backoffLastFailureTime = Date.now();

  if (backoffFailureCount === 1) {
    backoffCurrentDelay = BACKOFF_INITIAL_DELAY;
  } else {
    backoffCurrentDelay = Math.min(
      backoffCurrentDelay * BACKOFF_MULTIPLIER,
      BACKOFF_MAX_DELAY
    );
  }

  appendCentinelLog(
    request,
    `validator failure (${reason}); backoff=${backoffCurrentDelay}ms count=${backoffFailureCount}`
  );
}

function recordValidatorSuccess(request) {
  if (backoffFailureCount > 0) {
    appendCentinelLog(request, "validator success; backoff reset");
  }

  backoffFailureCount = 0;
  backoffLastFailureTime = 0;
  backoffCurrentDelay = 0;
}

function shouldSkipValidatorCall(request) {
  if (backoffFailureCount === 0) {
    return false;
  }

  const timeSinceLastFailure = Date.now() - backoffLastFailureTime;
  if (timeSinceLastFailure < backoffCurrentDelay) {
    appendCentinelLog(
      request,
      `skip validator (backoff) remaining=${backoffCurrentDelay - timeSinceLastFailure}ms`
    );
    return true;
  }

  appendCentinelLog(request, "backoff expired; retry validator");
  return false;
}

function base64ToBytes(str) {
  if (!str) return [];

  // Fast path if atob is available.
  if (typeof atob === "function") {
    const bin = atob(str);
    const bytes = new Array(bin.length);
    for (let i = 0; i < bin.length; i++) {
      bytes[i] = bin.charCodeAt(i) & 0xff;
    }
    return bytes;
  }

  // Fallback base64 decoder
  const chars = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/=";
  const bytes = [];
  let buffer = 0;
  let bits = 0;

  for (let i = 0; i < str.length; i++) {
    const idx = chars.indexOf(str.charAt(i));
    if (idx < 0) continue;
    if (idx === 64) break;

    buffer = (buffer << 6) | idx;
    bits += 6;

    if (bits >= 8) {
      bits -= 8;
      bytes.push((buffer >>> bits) & 0xff);
    }
  }

  return bytes;
}

function base64DecodeUtf8(str) {
  const bytes = base64ToBytes(str);
  if (!bytes || bytes.length === 0) return "";

  try {
    if (typeof TextDecoder === "function") {
      return new TextDecoder("utf-8").decode(new Uint8Array(bytes));
    }
  } catch (e) {}

  // Best-effort fallback (latin1-ish)
  let out = "";
  for (let i = 0; i < bytes.length; i++) {
    out += String.fromCharCode(bytes[i]);
  }
  return out;
}

function cookieObjectToHeaderString(cookie) {
  if (!cookie) return null;

  if (typeof cookie === "string") {
    return cookie;
  }

  if (typeof cookie !== "object") {
    return null;
  }

  const name = cookie.name;
  const value = cookie.value;

  if (!name || value == null) {
    return null;
  }

  const maxAge = cookie.max_age != null ? cookie.max_age : cookie.maxAge;
  const httpOnly = cookie.http_only != null ? cookie.http_only : cookie.httpOnly;
  const sameSite = cookie.same_site != null ? cookie.same_site : cookie.sameSite;

  let cookieStr = `${name}=${value}`;

  if (cookie.domain) cookieStr += `; Domain=${cookie.domain}`;
  cookieStr += `; Path=${cookie.path || "/"}`;
  if (cookie.expires) cookieStr += `; Expires=${cookie.expires}`;
  if (maxAge != null) cookieStr += `; Max-Age=${maxAge}`;

  // Default secure-ish attributes (Centinel cookies are session tracking)
  if (cookie.secure !== false) cookieStr += "; Secure";
  if (httpOnly !== false) cookieStr += "; HttpOnly";
  if (sameSite) cookieStr += `; SameSite=${sameSite}`;
  else cookieStr += "; SameSite=Lax";

  return cookieStr;
}

function storeFlowVariable(request, flow) {
  try {
    request.setVariable("PMUSER_EW_CENTINEL_FLOW", JSON.stringify(flow));
  } catch (e) {}
}

async function callCentinelValidator(request, payload, timeoutMs, secretKey) {
  const body = JSON.stringify(payload);

  const headers = {
    "Content-Type": "application/json",
    "x-api-key": secretKey,
  };

  const options = {
    method: "POST",
    headers,
    body,
    timeout: timeoutMs,
  };

  return await httpRequest(CENTINEL_INTERNAL_VALIDATE_PATH, options);
}

function getEnableLogging(request) {
  const v = request.getVariable("PMUSER_EW_CENTINEL_ENABLE_LOGGING");
  return v === "true";
}

export async function onClientRequest(request) {
  // Reset per-request state
  storeCentinelLogHeader(request, "");
  try {
    request.setVariable("PMUSER_EW_CENTINEL_FLOW", "");
  } catch (e) {}

  try {
    const { fullUrl, path } = parseRequestUrlParts(request);

    // Safety: never protect the internal validation route (prevents recursion)
    if (path === CENTINEL_INTERNAL_VALIDATE_PATH) {
      return;
    }

    const secretKey = request.getVariable("PMUSER_EW_CENTINEL_SECRET_KEY");
    if (!secretKey || String(secretKey).trim() === "") {
      appendCentinelLog(request, "CENTINEL secret key missing; disabled");
      return;
    }

    const inclusionPattern = request.getVariable(
      "PMUSER_EW_CENTINEL_PROTECTED_PATHS_INCLUSION"
    );
    const exclusionPattern = request.getVariable(
      "PMUSER_EW_CENTINEL_PROTECTED_PATHS_EXCLUSION"
    );

    const inclusionRe = toRegExp(inclusionPattern);
    const exclusionRe = toRegExp(exclusionPattern) || DEFAULT_PROTECTED_PATHS_EXCLUSION;

    if (!shouldProtectPath(path, inclusionRe, exclusionRe)) {
      appendCentinelLog(request, `skip validation (unprotected path) ${path}`);
      return;
    }

    // Backoff skip
    if (shouldSkipValidatorCall(request)) {
      return;
    }

    let timeoutMs = DEFAULT_TIMEOUT_MS;
    const timeoutVar = request.getVariable("PMUSER_EW_CENTINEL_TIMEOUT");
    if (timeoutVar != null) {
      const parsed = Number.parseInt(String(timeoutVar), 10);
      if (!Number.isNaN(parsed) && parsed >= 0) {
        timeoutMs = parsed;
      }
    }

    appendCentinelLog(request, `timeout=${timeoutMs}ms`);

    const headersObj = buildHeadersObjectExcludingCookie(request);
    const centinelCookie = getCentinelCookieValue(request);
    const clientIp = getBestClientIp(request);

    const payload = {
      url: fullUrl,
      method: request.method || "GET",
      ip: clientIp,
      referrer: getHeaderValue(request, "Referer"),
      headers: headersObj,
      cookie: centinelCookie,
    };

    let response;
    try {
      response = await callCentinelValidator(request, payload, timeoutMs, secretKey);
    } catch (e) {
      recordValidatorFailure(request, "httpRequest error");
      return;
    }

    if (!response || response.status !== 200) {
      recordValidatorFailure(request, `status ${response ? response.status : "null"}`);
      return;
    }

    let responseText = "";
    try {
      responseText = await response.text();
    } catch (e) {
      recordValidatorFailure(request, "read body");
      return;
    }

    let parsed;
    try {
      parsed = JSON.parse(responseText);
    } catch (e) {
      recordValidatorFailure(request, "json parse");
      return;
    }

    if (!parsed || typeof parsed !== "object" || !parsed.decision) {
      recordValidatorFailure(request, "missing decision");
      return;
    }

    recordValidatorSuccess(request);

    const decision = parsed.decision;
    appendCentinelLog(request, `decision=${decision}`);

    const cookies = Array.isArray(parsed.cookies) ? parsed.cookies : null;
    const cookieHeaders = cookies
      ? cookies.map(cookieObjectToHeaderString).filter((c) => c)
      : null;

    // Store flow for onClientResponse (cookies + decision)
    if (cookieHeaders && cookieHeaders.length > 0) {
      storeFlowVariable(request, {
        decision,
        cookies: cookieHeaders,
      });
    }

    if (decision === "allow" || decision === "not_matched") {
      return;
    }

    const responseHtmlB64 = parsed.response_html || "";
    let decodedHtml = "";

    try {
      decodedHtml = responseHtmlB64 ? base64DecodeUtf8(responseHtmlB64) : "";
    } catch (e) {
      decodedHtml = "";
    }

    if (!decodedHtml) {
      // No fallback HTML.
      // For explicit blocks, deny with an empty 403.
      // For redirects (challenge), fail open to avoid trapping users on a blank page.
      if (decision === "block") {
        request.respondWith(
          403,
          {
            "Content-Type": "text/html",
            "Cache-Control": "no-store",
          },
          ""
        );
        return;
      }

      recordValidatorFailure(request, "missing response_html");
      return;
    }

    const statusCode = decision === "redirect" ? 200 : 403;

    // Note: Set-Cookie is injected in onClientResponse (respondWith may not support it reliably).
    const responseHeaders = {
      "Content-Type": "text/html",
      "Cache-Control": "no-store",
    };

    request.respondWith(statusCode, responseHeaders, decodedHtml);
  } catch (error) {
    appendCentinelLog(request, `error onClientRequest=${String(error)}`);
    // Always fail open
  }
}

export function onClientResponse(request, response) {
  // Apply cookies if present
  const flow = request.getVariable("PMUSER_EW_CENTINEL_FLOW");
  if (flow) {
    try {
      const parsed = JSON.parse(flow);
      const cookies = parsed && parsed.cookies;

      if (cookies && Array.isArray(cookies) && cookies.length > 0) {
        for (let i = 0; i < cookies.length; i++) {
          const strCookie = cookies[i];
          if (!strCookie) continue;

          try {
            const cookie = new SetCookie(strCookie);
            response.addHeader("Set-Cookie", cookie.toHeader());
          } catch (e) {
            // If SetCookie fails, attempt to set raw value
            try {
              response.addHeader("Set-Cookie", strCookie);
            } catch (e2) {}
          }
        }
      }
    } catch (e) {}
  }

  // Debug header (optional)
  try {
    if (getEnableLogging(request)) {
      response.addHeader(
        "X-Centinel-Log",
        request.getVariable("PMUSER_EW_CENTINEL_LOG_HEADER")
      );
    }
  } catch (e) {}
}
